/************************************************************************
 * @file: AudioFader.cpp
 *
 * @description: This header file contains implementation for class Fader.
 * Fader is a Post Processing utility class for applying FADE_IN/FADE_OUT on
 * PCM data. Fader class will be used by Backend libraries for applying Fading
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Thouseef Ahamed, tahamed@de.adit-jv.com 2015
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *           Vanitha Channaiah, vanitha.channaiah@in.bosch.com 2016
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <endian.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <errno.h>
#include <math.h>
#include <string>
#include <cstring>

#include "AudioFader.h"

using namespace adit::utility::audio;

#define RES_SHIFT 16
#define VOL_RANGE (1 << RES_SHIFT)

/* Level offset for 8bit data (128) and 16bit data (32768)*/
#define U8_OFFSET ((1 << 8) >> 1)
#define U16_OFFSET ((1 << 16) >> 1)

#define SWAP(x) SwapBytes(&x, sizeof(x));

/* TODO:
 * Need to support Exponential Ramp calculation and processing
 */

#if __BYTE_ORDER == __LITTLE_ENDIAN
#define FADER_LITTLE_ENDIAN
#elif __BYTE_ORDER == __BIG_ENDIAN
#define FADER_BIG_ENDIAN
#else
#error "Unsupported endian..."
#endif

template<typename E>
void Fader::SwapBytes(E *value, unsigned int size)
{
    char *p = (char *)value;
    unsigned int i, j;
    if (mSwap == TRUE)
    {
        for(i = 0, j = size-1; j > i; i++, j--)
        {
            char tmp = p[i];
            p[i] = p[j];
            p[j] = tmp;
        }
    }
}

Fader::Fader(LoggingInterface& loggingHandle) : mLoggingHandle(loggingHandle)
{
#ifdef FADER_LITTLE_ENDIAN
            mIsLe = TRUE;
#else
            mIsLe = FALSE;
#endif
    mFadeoutFrames = 0;
    mFadeFrames = 0;
    mlevelOffset = 0;
    mDir = FaderDir::OFF;
    mFormat = UNKNOWN;
    mState = FADER_STATE_STOPPED;
    mSwap = FALSE;
    mCurve = nullptr;
    defaults();
}

Fader::~Fader()
{
    if (mCurve != nullptr)
    {
        delete [] mCurve;
    }
}

FaderError Fader::reset()
{
    mState = FADER_STATE_STOPPED;
    mRemainingFrames = 0;
    mFadeoutFrames = 0;
    mFadeFrames = 0;
    defaults();
    Logging(mLoggingHandle, LL_DEBUG) << "Fader::reset() with the state: " << mState << Logging::endl;
    return FaderError::OK;
}

void Fader::defaults()
{
    mRemainingFrames = 0;
    mFadeTime = 0;
}

FaderError Fader::isFormatSupported(const AudioFormat format)
{
    mFormat = format;
    FaderError ret = FaderError::OK;
    switch (format) {
        case S8:
            break;
        case U8:
            mlevelOffset = U8_OFFSET;
            break;
        case S16_BE:
            mSwap = mIsLe;
            break;
        case U16_BE:
            mSwap = mIsLe;
            mlevelOffset = U16_OFFSET;
            break;
        case S16_LE:
            mSwap = !mIsLe;
            break;
        case U16_LE:
            mSwap = !mIsLe;
            mlevelOffset = U16_OFFSET;
            break;
        default :
            ret = FaderError::UNSUPPORTED;
            break;
    }
    return ret;

}

template<typename Func>
FaderError Fader::calcRamp(Func func)
{
    /* free the curveLin memory if configure is called multiple times */
    if (mCurve != nullptr)
    {
        delete [] mCurve;
        mCurve = nullptr;
    }

    if (mFadeFrames != 0)
    {
        mCurve = new uint32_t[mFadeFrames];
        if (!mCurve)
        {
            return FaderError::FAILURE;
        }
    }

    for (uint32_t i = 0; i < mFadeFrames; i++)
    {
        func (i);
    }

    return FaderError::OK;
}

FaderError Fader::calcLinearRamp()
{
    auto linearFunction = [this] (uint32_t i)
    {
        mCurve[i] = VOL_RANGE - ((static_cast<int64_t>(i) * VOL_RANGE) / mFadeFrames);

    };
    return calcRamp<> (linearFunction);
}

FaderError Fader::configure(const AudioFormat format, const uint32_t rate, const uint32_t channels)
{
    if (mState == FADER_STATE_PROCESSING)
    {
        return FaderError::FAILURE;
    }

    if (mDir == FaderDir::OFF)
    {
        return FaderError::FAILURE;
    }

    /* Unsupported formats returns error */
    if (isFormatSupported(format) != FaderError::OK)
    {
        Logging(mLoggingHandle, LL_DEBUG) << "Fader::configure() Failed with unsupported formats: " << format << Logging::endl;
        return FaderError::UNSUPPORTED;
    }

    if (channels == 0)
    {
        return FaderError::UNSUPPORTED;
    }

    mRate          = rate;
    mChannels      = channels;

    Logging(mLoggingHandle, LL_DEBUG) << "Fader::configure() format = " << format << "  rate = " << rate << "  channels = " << channels << Logging::endl;

    return doConfigure();
}

FaderError Fader::doConfigure()
{
    mFadeFrames  = (mFadeTime * mRate) / 1000;
    mRemainingFrames = mFadeFrames;

    FaderError ret = calcLinearRamp();
    if (ret == FaderError::OK)
    {
        mState = FADER_STATE_CONFIGURED;
    }

    Logging(mLoggingHandle, LL_DEBUG) << "Fader::doConfigure() with the state: " << mState << Logging::endl;

    return ret;
}

FaderError Fader::restart()
{
    if (mState == FADER_STATE_STOPPED)
    {
        return FaderError::FAILURE;
    }

    mRemainingFrames = mFadeFrames;
    mFadeoutFrames = 0;
    return FaderError::OK;
}

FaderError Fader::setFadeDirectionAndTime(const FaderDir direction, const uint32_t fadeTime)
{
    if (direction == FaderDir::OFF)
    {
        return FaderError::NOTCONFIGURED;
    }

    mDir = direction;
    mFadeTime = fadeTime;
    return FaderError::OK;
}

FaderError Fader::process(const void *buf, uint32_t frames, void *outbuff)
{
    FaderError ret;

    if ((outbuff == NULL) || (buf == NULL))
    {
        return FaderError::FAILURE;
    }

    if (mState == FADER_STATE_STOPPED)
    {
        return FaderError::NOTCONFIGURED;
    }

    switch (mFormat)
    {
        case S8:
            ret = processRampData(static_cast<const int8_t*>(buf), &mCurve[0], frames, static_cast<int8_t*>(outbuff));
            break;
        case U8:
            ret = processRampData(static_cast<const uint8_t*>(buf), &mCurve[0], frames, static_cast<uint8_t*>(outbuff));
            break;
        case S16_LE:
        case S16_BE:
            ret = processRampData(static_cast<const int16_t*>(buf), &mCurve[0], frames, static_cast<int16_t*>(outbuff));
            break;
        case U16_LE:
        case U16_BE:
            ret = processRampData(static_cast<const uint16_t*>(buf), &mCurve[0], frames, static_cast<uint16_t*>(outbuff));
            break;
        default :
            ret = FaderError::UNSUPPORTED;
            break;
    }

    if (ret == FaderError::OK)
    {
        mState = FADER_STATE_PROCESSING;
        if (mRemainingFrames == 0)
        {
            ret = FaderError::DONE;
        }
    }
    return ret;
}

uint32_t Fader::getRemainingFrames()
{
    return mRemainingFrames;
}

FaderError Fader::getFadeTime(uint32_t& fadeTime)
{
    fadeTime = mFadeTime;

    return FaderError::OK;
}

template<typename T>
FaderError Fader::processRampData(const T* buf, uint32_t* rampAddr, const uint32_t frames, T *output)
{
    uint32_t level = 0;
    uint32_t ch    = 0;
    uint32_t i     = 0;
    uint32_t remainFrames;
    T invalue;

    remainFrames = (mRemainingFrames <= frames) ?  mRemainingFrames : frames;

    /* Applying ramp for fade-in or fade-out */
    for (i = 0; i < remainFrames; i++ )
    {
        level = (mDir == FaderDir::IN) ? rampAddr[mRemainingFrames - i - 1] : rampAddr[mFadeoutFrames + i];

        for (ch = 0; ch < mChannels; ch++)
        {
            invalue = *buf;
            SWAP(invalue);
            *output = (((static_cast<int64_t>(level) * (invalue - mlevelOffset)) / VOL_RANGE) + mlevelOffset);
            SWAP(*output);
            buf++;
            output++;
        }
    }
    mRemainingFrames -= remainFrames;
    mFadeoutFrames += remainFrames;

    /* Input samples exceeding fade time, fade-in will not be applied */
    if (((frames - remainFrames) > 0) && (mDir == FaderDir::IN))
    {
        /* copy input data to output */
        memcpy(output, buf, ((frames - remainFrames) * mChannels * sizeof(T)));
    }
    /* Input samples exceeding fade time, are made silence */
    else if ((static_cast<int64_t>(frames - mFadeoutFrames) > 0) && (mDir == FaderDir::OUT))
    {
        for (i = 0; i < (frames - remainFrames); i++)
        {
            for (ch = 0; ch < mChannels; ch++)
            {
                *output++ = mlevelOffset;
            }
        }
    }
    return FaderError::OK;
}
